/*
 * @(#)ANIMApplet.java  3.3  2006-10-01
 *
 * Copyright (c) 1999-2006 Werner Randelshofer
 * Staldenmattweg 2, CH-6405 Immensee, Switzerland
 * All rights reserved.
 *
 * This software is the confidential and proprietary information of
 * Werner Randelshofer. ("Confidential Information").  You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Werner Randelshofer.
 */


package ch.randelshofer.app;

import java.applet.Applet;
import java.awt.*;
import java.awt.image.*;
import java.net.URL;
import java.net.URLConnection;
import java.io.InputStream;
import java.io.IOException;
import java.util.zip.ZipInputStream;
import java.util.zip.ZipEntry;
import java.util.*;
import ch.randelshofer.gui.*;
import ch.randelshofer.io.*;
import ch.randelshofer.media.*;
import ch.randelshofer.media.anim.*;
import ch.randelshofer.util.*;
import java.beans.*;

/**
 * This Applet plays files of type Amiga IFF ANIM.
 *
 * @author Werner Randelshofer, Staldenmattweg 2, CH-6405 Immensee, Switzerland
 * @version 3.3 2006-10-01 Added support for Eric Graham's compression method 
 * and for ping-poing playback.
 * <br>3.2 2005-07-14 Changed to another threading model in the player.
 * Fall back to imageSource, imageArchive parameters if animSource, animArchive
 * parameters are not present. Parameter "preferTrueColor" added. Improved
 * behaviour of slider.
 * <br>3.0.2 2005-02-20 Use different values for Jiffies depending on the
 * Commodore Amiga Graphics mode (CAMG).
 * <br>3.0.1 2005-01-10 Try harder to figure out an acceptable value for
 * imageFileSize.
 * <br>3.0 2004-12-28 Renamed applet paramaters, because they interfered
 * with the special applet parametes for the Java plugin from Sun.
 * <br>2.1.1 2004-12-23 Use nearest neighbor interpolation when scaling
 * image.
 * <br>2.1 2004-08-21 Minor bug fix in class AbstractButton. Improved
 * performance of class Bitmap.
 * <br>2.0.1 Minor bug fixes. Parameters 'debug', 'info' and 'audio' added.
 * <br>2003-04-31 MovieControl improved. Support for animations with sound
 * (ANIM+SLA and ANFI) added.
 * <br>1.0.1 2003-03-31 IFFParsers uses now MC68000InputStream.skipFully to
 * skip input data instead of MC68000InputStream.skip. Changed package names.
 * <br>1.0  	1999-10-11	Minor improvements on MovieSlider.
 *				Image was cut by one pixel on rigth and bottom.
 * <br>history	0.4  	1999-05-30	Added support for more applet parameters.
 * <br>history	0.3  	1999-05-09	Support for ZIP files added.
 * <br>history	0.2.2 1999-05-09  Many bugs fixed.
 * <br>history	0.1   1999-05-01  Created.
 */
public class ANIMApplet
extends Applet
implements Runnable, PropertyChangeListener {
    private final static String version = "3.4 2008-04-05";
    
    /**
     * Movie Controller.
     */
    private MovieControlAqua controllerPanel;
    /**
     * Image Panel.
     */
    private ImagePanelAWT imagePanel;
    
    /**
     * ANIM Player.
     */
    private ANIMPlayer  player;
    
    /**
     * Background thread loads animation file.
     */
    private Thread loaderThread;
    
    /**
     * This flag indicates whether the animation was playing
     * before the applet has been stopped.
     */
    private boolean isPlaying;
    
    /**
     * Array of message Strings used to display error messages.
     */
    private String[] messages;
    
    /**
     * This wrapper for the input stream makes it easy to suspend
     * and resume the loader thread when the applet is being stopped
     * or started by the browser.
     */
    private SuspendableInputStream suspendableInputStream;
    
    /**
     * Set this to true to turn debugging on.
     */
    private boolean debug;
    
    /**
     * Initializes the applet.
     */
    public void init() {
        String param;
        
        debug = Applets.getParameter(this, "debug", false);
        
        // Layout: ImagePanel at center, MovieController at south.
        setLayout(new BorderLayout());
        
        // Set the background color according to the 'backgroundColor' parameter.
        setBackground(Applets.getParameter(this, "boxBgColor", Color.white));
        
        // Set the foreground color according to the 'foregroundColor' parameter.
        setForeground(Applets.getParameter(this, "boxFgColor", Color.black));
    }
    /**
     * Paints a message on the display area of
     * the applet. This message is only visible during
     * initialisation of the applet. After that it is
     * obscured by the components of the applet.
     */
    public void paint(Graphics g) {
        FontMetrics fm = g.getFontMetrics();
        int strAscent = fm.getAscent();
        int strHeight = fm.getHeight();
        if (messages == null) {
            g.drawString("Loading ANIMApplet "+version,0,strAscent);
            g.drawString("\u00a9 Werner Randelshofer",0,strAscent+strHeight);
        } else {
            if (messages != null) {
                int y = 0;
                for (int i=0; i < messages.length; i++) {
                    int strWidth = fm.stringWidth(messages[i]);
                    g.drawString(messages[i],0, y + strAscent);
                    y += strHeight;
                }
            }
        }
    }
    
    /**
     * Loads the animation file.
     * This method may be called from the loader thread only.
     *
     * XXX This method should not be public because it exposes internal
     *	behaviour of the applet.
     */
    public void run() {
        try {
            String imageSource = Applets.getParameter(this, "animSource", Applets.getParameter(this, "imageSource", ""));
            String imageArchive = Applets.getParameter(this, "animArchive", Applets.getParameter(this, "imageArchive", (String) null));
            String image = Applets.getParameter(this, "anim", Applets.getParameter(this, "image", (String) null));
            InputStream in;
            int imageFileSize;
            ZipEntry zipEntry;
            BoundedRangeInputStream bris;
            ANIMDecoder decoder;
            
            if (imageSource.length() > 0 && imageSource.charAt(imageSource.length()-1) != '/') {
                imageSource = imageSource+'/';
            }
            
            // Is the animation stored in a ZIP file?
            if (imageArchive != null) {
                // ...create input stream from ZIP entry.
                URL url = new URL(getDocumentBase(), imageSource+imageArchive );
                URLConnection connection = url.openConnection();
                connection.connect();
                suspendableInputStream = new SuspendableInputStream(connection.getInputStream());
                ZipInputStream zipIn = new ZipInputStream(suspendableInputStream) {
                    // XXX This inner class is a workaround for bugs in
                    // the skip-method of class java.util.zip.ZipInputStream
                    // from JDK 1.1
                    public long skip(long n) throws IOException {
                        long total = 0;
                        while (total < n) {
                            long len = super.skip( Math.min(n - total,512) );
                            if (len == -1) {
                                if (total == 0) {
                                    total = -1;
                                }
                                break;
                            }
                            total += len;
                        }
                        return total;
                    }
                };
                
                // Seek for our animation file.
                do {
                    zipEntry = zipIn.getNextEntry();
                } while (! zipEntry.getName().equals(image) );
                
                
                // In order to display a progress bar, we need to know the
                // image file size. If possible, we attempt to get this value
                // from the ZipEntry. If this value is not available, we use
                // the size of the Zip File multiplied by two as a substitute.
                // We use factor two, because this appears to be a reasonable
                // assumption for the compression achievable by deflate for
                // ANIM files.
                // We display a warning message on the console, so that
                // we now, when the Applet does not use an accurate value for
                // the image file size.
                imageFileSize = (int) zipEntry.getSize();
                if (imageFileSize == -1) {
                    System.err.println("ANIMApplet Warning:"
                    +"\n  Unable to show an accurate progress bar while loading the ANIM file."
                    +"\n  Please recompress the Zip archive with a tool that stores the file size at the beginning of the Zip entries."
                    +"\n  Zip Archive: "+url
                    );
                    imageFileSize = connection.getContentLength();
                    if (imageFileSize != -1) {
                        imageFileSize *= 2;
                    }
                }
                
                in = zipIn;
                
                // Is the animation not stored in a ZIP file?
            } else {
                // ...create input stream from URL.
                URL url = new URL(getDocumentBase(), imageSource + image);
                URLConnection connection = url.openConnection();
                connection.connect();
                imageFileSize = connection.getContentLength();
                in = suspendableInputStream = new SuspendableInputStream(connection.getInputStream());
            }

            // Create the player from the animation file.
            player = new ANIMPlayer(in, imageFileSize, Applets.getParameter(this, "audio", true));
            player.debug = debug;
            
            if (Applets.getParameter(this,"renderAsTrueColor",true)) {
                player.setPreferredColorModel(DirectColorModel.getRGBdefault());
            }
            
            String p;
      
            //player.addPropertyChangeListener(this);
            controllerPanel = (MovieControlAqua) player.getControlPanelComponent();
            // Set the settings for the controller panel
            if (Applets.getParameter(this, "controller", true)) {
                add(BorderLayout.SOUTH,controllerPanel);
            }
            
            imagePanel = (ImagePanelAWT) player.getVisualComponent();
            // Set the scale settings for the image panel
            p = Applets.getParameter(this, "scale","toFit").toLowerCase();
            if (p.equals("tofit") ) {
                imagePanel.setImageScalePolicy(ImagePanelAWT.SCALE_TO_FIT);
            } else if (p.equals("aspect") ) {
                imagePanel.setImageScalePolicy(ImagePanelAWT.SCALE_TO_ASPECT);
            } else {
                imagePanel.setImageScalePolicy(ImagePanelAWT.SCALE_TO_SIZE);
                try {
                    double factor = Double.valueOf(p).doubleValue();
                    imagePanel.setScaleFactor(factor,factor);
                } catch (NumberFormatException e) {
                    imagePanel.setMessage("Invalid property: scale="+p);
                    imagePanel.setImageScalePolicy(ImagePanelAWT.SCALE_TO_FIT);
                }
            }
            add(BorderLayout.CENTER,imagePanel);
            
            StateTracker tracker = new StateTracker(player);
            player.realize();
            int[] states = { Player.REALIZED, Player.CLOSED };
            tracker.waitForState(states);
            tracker.setStateModel(null);
            
            player.setLoop(Applets.getParameter(this, "loop", true));
            player.setPingPong(Applets.getParameter(this, "pingPong", false));
            
            player.getMovieTrack().setPlayWrapupFrames(Applets.getParameter(this, "playWrapupFrames", false));
            
            p = getParameter("framesPerSecond");
            if (p != null) {
                try {
                    player.setFramesPerSecond(Float.valueOf(p).floatValue());
                } catch (NumberFormatException e) { // XXX maybe this is a good place to show an error message
                }
            }
            
            doLayout();
            repaint();
            
            player.addPropertyChangeListener(this);
            if (Applets.getParameter(this, "autoStart", true)) {
                if (player.isCached()) player.start();
            }
        } catch (Throwable e) {
            setMessage(e.toString()+"\n"+e.getMessage());
            removeAll();
            doLayout();
            repaint();
            e.printStackTrace();
        }
    }
    
    
    
    /**
     * Starts the applet.
     */
    public void start() {
        // Create the loader thread and start it.
        if (loaderThread == null) {
            loaderThread = new Thread(this);
            try {
                loaderThread.setPriority(Thread.MIN_PRIORITY);
            } catch (Throwable e) {
                // Some browsers don't allow us to set the priority of a Thread.
            }
            loaderThread.start();
        }
        
        // Resume the loader thread.
        if (suspendableInputStream != null) {
            suspendableInputStream.resume();
        }
        
        // Resume the player thread.
        if (isPlaying == true && player != null) {
            player.start();
        }
    }
    
    /**
     * Stops the applet.
     */
    public void stop() {
        // Suspend the loader thread.
        if (suspendableInputStream != null) {
            suspendableInputStream.suspend();
        }
        
        // Suspend the player thread.
        if (player != null) {
            isPlaying = player.isActive();
            player.stop();
        }
    }
    
    /**
     * Destroys the applet.
     */
    public void destroy() {
        // Destroy the loader thread.
        if (suspendableInputStream != null) {
            suspendableInputStream.abort();
        }
        
        try {
            if (loaderThread != null) 	{
                loaderThread.interrupt();
                loaderThread = null;
            }
        } catch (Throwable t) {}
        
        // Destroy the player thread.
        if (player != null) {
            player.stop();
            player = null;
        }
    }
    
    /**
     * Get applet info for HTML authoring tools.
     */
    public String getAppletInfo() {
        return "ANIMApplet "+version+" \u00a9 "+
                "1999-2005 Werner Randelshofer, Staldenmattweg 2, CH-6405 Immensee, Switzerland";
    }
    
    /**
     * Get applet parameter info for HTML authoring tools.
     */
    public String[][] getParameterInfo() {
        return new String[][] {
            {"animSource",  "URL",  "Anim directory."},
            {"animArchive",  "URL",  "Anim zip file."},
            {"anim",    "URL",  "Anim file"},
            {"boxMessage",    "String",  "Text shown in status bar while applet is being loaded."},
            {"boxBgColor",    "Color",  "Background color, e.g. '#ffffff', '0xffffff', 'white', or '255,255,255'. Default depends on browser settings."},
            {"boxFgColor",    "Color",  "Foreground color, e.g. '#000000', '0x000000', 'black', or '0,0,0'. Default depends on browser settings."},
            {"renderAsTrueColor",    "boolean",  "Renders the animation using a 24 bit direct color model if set to 'true'. Default value is 'true'."},
            {"image",    "URL",  "Image shown while applet is being loaded."},
            {"scale", "float|'toFit'|'aspect'", "float: scale factor e.g. '2.0'=doubles the size. 'toFit' scales the animation to the bounding box of the view area. 'aspect' scales to fit while maintaining the aspect ratio. Default is 'tofit'." },
            {"audio", "boolean", "'true' enables audio playback. Default is 'true'." },
            //	{"colorCycling", "boolean", "'true' turns color cycling on. Default is 'true'." }
            {"framesPerSecond",    "float",  "Number of frames per second. '0' use animation settings. Default is '0'."},
            {"playEveryFrame",    "boolean",  "'true' plays every frame of the animation. Default is 'false'."},
            {"playWrapupFrames", "boolean", "'true' treats wrapup frames like regular frames. Default is 'false'." },
            {"controller", "boolean", "'true' shows the controller. Default is 'true'." },
            {"autoPlay", "boolean", "'true' starts playing after the animation has been loaded. Default is 'true'." },
            {"loop",    "boolean",  "'true' makes the animation play continuously. Default is 'true'."},
            {"pingPong",    "boolean",  "'true' makes the animation play forward and backwards. Default is 'false'."},
            {"info", "boolean", "'true' Displays information about the animation. Default is 'false'" },
            {"debug", "boolean", "'true' turns output of debugging info on. Default is 'false'." },
        };
    }
    
    public void propertyChange(PropertyChangeEvent evt) {
        if (evt.getPropertyName().equals("cached")) {
            if (Applets.getParameter(this, "autoStart", true)) {
                player.start();
            }
            if (Applets.getParameter(this, "info", false)) {
                ANIMMovieTrack track = player.getMovieTrack();
                double aspectRatio = track.getXAspect() / (double) track.getYAspect();
                Dimension correctedImageSize = new Dimension(track.getWidth(), track.getHeight());
                if (aspectRatio > 1f) {
                    correctedImageSize.width = (int) Math.floor(track.getWidth() * aspectRatio);
                } else {
                    correctedImageSize.height = (int)Math.ceil(track.getHeight()/aspectRatio);
                }
                
                Dimension size = getSize();
                
                String msg =
                "Movie"
                +"\n  Duration: "+(player.getTotalDuration() / 60f)+"sec";
                if (getParameter("framesPerSecond") != null) {
                    msg += " ("+(track.getTotalDuration() / 60f)+"sec)";
                    }
                msg += "\n  Jiffies: "+track.getJiffies();
                msg += "\n  Frame Count: "+track.getFrameCount();
                if (Applets.getParameter(this, "playWrapupFrames", false))
                    msg += " ("+(track.getFrameCount() - 2)+")";
                if (track.getProperty("copyright") != null)
                    msg += "\n  Copyright: \""+track.getProperty("copyright")+"\"";
                if (track.getProperty("author") != null)
                    msg += "\n  Author: \""+track.getProperty("author")+"\"";
                if (track.getProperty("annotation") != null)
                    msg += "\n  Annotation: \""+track.getProperty("annotation")+"\"";
                msg += "\nImage"
                +"\n  Screen Mode: "+imagePanel.getImage().getProperty("screenMode", this)
                +"\n  Plane Count: "+track.getNbPlanes()
                +"\n  Pixel Aspect x:"+track.getXAspect()    + " y:"+track.getYAspect()
                +"\n  Size w:"+size.width+" h:"+size.height+" (w:"+track.getWidth() + " h:"+track.getHeight()+")"
                +"\n  Aspect Corrected Size w:"+correctedImageSize.width+" h:"+correctedImageSize.height;
                //+"\nViewer Dimension w:"+imagePanel.getWidth() + " h:"+imagePanel.getHeight()
                if (track.getAudioClipCount() > 0)
                    msg += "\n \nSound"
                    +"\n  Clip Count: "+track.getAudioClipCount();
                msg += "\nANIMApplet "+version
                +"\n  Preferred Size w:"+getPreferredSize().width+" h:"+getPreferredSize().height;
                
                
                imagePanel.setMessage(msg);
            }
        }
    }
    
    public void setMessage(String message) {
        Vector v = new Vector();
        StringTokenizer tt = new StringTokenizer(message, "\n");
        while (tt.hasMoreTokens()) {
            v.addElement(tt.nextToken());
        }
        this.messages = new String[v.size()];
        v.copyInto(this.messages);
        repaint();
    }
}
