/*
 * @(#)ImagePanelAWT.java  1.2  2004-12-23
 *
 * Copyright (c) 1999-2004 Werner Randelshofer
 * Staldenmattweg 2, CH-6405 Immensee, Switzerland
 * All rights reserved.
 *
 * This software is the confidential and proprietary information of
 * Werner Randelshofer. ("Confidential Information").  You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Werner Randelshofer.
 */
package ch.randelshofer.gui;

import java.awt.*;
import java.awt.image.*;
import java.awt.event.*;
import java.io.*;
import java.lang.reflect.*;
import java.net.*;
import java.util.*;
import java.beans.PropertyChangeSupport;
import java.beans.PropertyChangeListener;

/**
 * Displays images in a panel.
 *
 * @author	Werner Randelshofer, Staldenmattweg 2, CH-6405 Immensee, Switzerland.
 * @version	1.2 2004-12-23 Set the interpolate_nearest_neighbor rendering
 * hint, if Java2D is available.
 * <br>1.1 2003-04-25 Support for multi-line messages added.
 * <br>1.0  	1999-10-12	One line of pixels was clipped at the right and at the bottom
 * of the image.
 * Images with SCALE_TO_SOZE or SCALE_TO_IMAGE_ASPECT were not
 * drawn on Netscape Communicator 4.6/Navigator 4.08
 * for Macintosh.
 * <br>0.4.2	1999-05-30	SCALE_TO_VIEW_SIZE renamed to SCALE_TO_FIT.
 * SCALE_TO_IMAGE_SIZE renamed to SCALE_TO_SIZE.
 * SCALE_TO_IMAGE_ASPECT renamed to SCALE_TO_ASPECT.
 * <br>0.4.1	1999-05-02	Better compatibility with Netscape Navigator.
 * <br>0.4  	1999-04-05	#setPixelAspectPolicy throws IllegalArgumentException
 * when passing invalid arguments. HARDSCALED_PIXEL_ASPECT
 * renamed into MULTIPLIED_PIXEL_ASPECT. Method #getPixelAspectPolicy
 * added.
 * <br>0.3  	1999-03-09	#getScaledImageSize waits for the image to
 * be loaded.
 * <br>0.2  	1999-02-04	New Method #setMessage allows the specification
 * of a message text that will displayed on the
 * display area..
 * <br>0.1  	1999-02-02	First release.
 * <br>0.0 	1999-01-03	Created.
 */
public class ImagePanelAWT
extends Canvas {
    /**
     * Pixel aspect policy: Ignores pixel aspect.
     */
    public final static int IGNORE_PIXEL_ASPECT = 0;
    /**
     * Pixel aspect policy: Considers only extreme pixel aspects,
     * where one dimension is a multiple of the other dimension
     * of the pixel.
     */
    public final static int MULTIPLIED_PIXEL_ASPECT = 1;
    /**
     * Pixel aspect policy: Consider the exact pixel aspect.
     */
    public final static int EXACT_PIXEL_ASPECT = 2;
    
    /**
     * Image scaling policy: Scale as defined by #setScaleFactor
     * and the image aspect.
     */
    public final static int SCALE_TO_SIZE = 0;
    /**
     * Image scaling policy: Scale to panel size.
     */
    public final static int SCALE_TO_FIT = 1;
    /**
     * Image scaling policy: Scale to panel but keep the
     * image aspect.
     */
    public final static int SCALE_TO_ASPECT = 2;
    
    private Image image_;
    private double scaleX_ = 1.0;
    private double scaleY_ = 1.0;
    private int imageScalePolicy_ = SCALE_TO_FIT;
    private int pixelAspectPolicy_ = EXACT_PIXEL_ASPECT;
    private boolean keepExactPixelAspect_ = false;
    private String[] messages;
    
    /** Support for listeners. */
    private PropertyChangeSupport propertyChangeSupport_ = new PropertyChangeSupport(this);
    private Image offImage_;
    private Dimension offDimension_;
    private Graphics offGraphics_;
    
    public ImagePanelAWT() {
        super();
    }
    
    public void update(Graphics g) {
        paint(g);
    }
    
    
    public void paint(Graphics g) {
        Dimension size = getSize();
        if (image_ == null || image_.getWidth(this) == -1) {
            g.setColor(getBackground());
            g.fillRect(0, 0, size.width, size.height);
        } else {
            configure(g);
            final int iw = image_.getWidth(this);
            final int ih = image_.getHeight(this);
            double xAspect = getPixelAspectX();
            double yAspect = getPixelAspectY();
            
            switch (imageScalePolicy_) {
                case SCALE_TO_FIT : {
                    setScaleFactor0(size.width / (double)iw, size.height / (double)ih);
                    //drawCheckerboard(g,new Rectangle(0,0,size.width,size.height));
                    try {
                        g.drawImage(
                        image_,
                        0,0,size.width,size.height,
                        0,0,iw,ih,
                        getBackground(),
                        this
                        );
                    } catch (NoSuchMethodError e) {
                        g.drawImage(
                        image_,
                        0, 0, size.width, size.height,
                        getBackground(),
                        this
                        );
                    }
                    break;
                }
                case SCALE_TO_ASPECT : {
                    double factor = Math.min(size.width / (double)(iw*xAspect), size.height / (double)(ih*yAspect));
                    setScaleFactor0(factor,factor);
                    int w = (int)Math.ceil(iw*xAspect*factor);
                    int h = (int)Math.ceil(ih*yAspect*factor);
                    int x = (size.width - w)/2;
                    int y = (size.height - h)/2;
                    //drawCheckerboard(g,new Rectangle(x,y,w,h));
                    try {
                        g.drawImage(
                        image_,
                        x,y,x+w,y+h,
                        0,0,iw,ih,
                        getBackground(),
                        this
                        );
                    } catch (NoSuchMethodError e) {
                        g.drawImage(
                        image_,
                        x, y, w, h,
                        getBackground(),
                        this
                        );
                    }
                    if (size.width > w) {
                        g.setColor(getForeground());
                        g.drawLine(x-1, 0, x-1, size.height-1);
                        g.drawLine(x+w, 0, x+w, size.height-1);
                        g.setColor(getBackground());
                        g.fillRect(0, 0, x-1, size.height);
                        g.fillRect(x+w+1, 0, size.width, size.height);
                    }
                    else if (size.height > h) {
                        g.setColor(getForeground());
                        g.drawLine(0, y-1, size.width-1, y-1);
                        g.drawLine(0, y+h, size.width-1, y+h);
                        g.setColor(getBackground());
                        g.fillRect(0, 0, size.width, y-2);
                        g.fillRect(0,y+h+1, size.width, size.height);
                    }
                    break;
                }
                case SCALE_TO_SIZE : {
                    int w = (int)(iw*xAspect*getScaleFactorX());
                    int h = (int)(ih*yAspect*getScaleFactorY());
                    int x = (size.width - w)/2;
                    int y = (size.height - h)/2;
                    //drawCheckerboard(g,new Rectangle(x,y,w,h));
                    try {
                        g.drawImage(
                        image_,
                        x,y,x+w,y+h,
                        0,0,iw,ih,
                        getBackground(),
                        this
                        );
                    } catch (NoSuchMethodError e) {
                        g.drawImage(
                        image_,
                        x, y, w, h,
                        getBackground(),
                        this
                        );
                    }
                    g.setColor(getBackground());
                    if (size.width > w) {
                        g.fillRect(0, 0, x-1, size.height-1);
                        g.fillRect(x+w+1, 0, size.width-1, size.height-1);
                    }
                    if (size.height > h) {
                        g.fillRect(0, 0, size.width-1, y-1);
                        g.fillRect(0,y+h+1, size.width-1, size.height-1);
                    }
                    g.setColor(getForeground());
                    g.drawRect(x-1,y-1,w+1,h+1);
                    break;
                }
            }
        }
        String[] m = messages;
        if (m != null) {
            FontMetrics fm = g.getFontMetrics();
            int strAscent = fm.getAscent();
            int strHeight = fm.getHeight();
            int y = 0;
            for (int i=0; i < m.length; i++) {
                int strWidth = fm.stringWidth(m[i]);
                g.setColor(getBackground());
                g.fillRect(0, y, strWidth, strHeight);
                g.setColor(getForeground());
                g.drawString(m[i],0, y + strAscent);
                y += strHeight;
            }
        }
    }
    /**
     * Sets the pixel aspect policy.
     *
     * @param	policy	PixelAspectPolicy must be one of IGNORE_PIXEL_ASPECT,
     * MULTIPLIED_PIXEL_ASPECT, EXACT_PIXEL_ASPECT.
     *
     * @exception	IllegalArgumentException	When passing invalid policy.
     */
    public synchronized void setPixelAspectPolicy(int policy) {
        if (policy != IGNORE_PIXEL_ASPECT &&
        policy != MULTIPLIED_PIXEL_ASPECT &&
        policy != EXACT_PIXEL_ASPECT) {
            throw new IllegalArgumentException("Invalid policy:"+policy);
        }
        pixelAspectPolicy_ = policy;
        validate();
        repaint();
    }
    /**
     * Returns the pixel aspect policy.
     */
    public int getPixelAspectPolicy() {
        return pixelAspectPolicy_;
    }
    
    /**
     * Sets the image scale policy.
     */
    public synchronized void setImageScalePolicy(int policy) {
        if (policy != SCALE_TO_SIZE &&
        policy != SCALE_TO_FIT &&
        policy != SCALE_TO_ASPECT) {
            throw new IllegalArgumentException("Invalid policy:"+policy);
        }
        imageScalePolicy_ = policy;
        validate();
        repaint();
    }
    /**
     * Returns the image scale policy.
     */
    public int getImageScalePolicy() {
        return imageScalePolicy_;
    }
    
    /**
     * Gets the horizontal pixel aspect of the image according to the
     * pixel aspect policy that is in affect.
     *
     * @return	Horizontal pixel aspect.
     */
    public double getPixelAspectX() {
        if (image_ == null)
        { return 1; }
        
        Object property = image_.getProperty("aspect",this);
        if (property == null)
        { return 1; }
        double ratio = (property == null || property == Image.UndefinedProperty) ? 1d : ((Double)property).doubleValue();
        
        switch (pixelAspectPolicy_) {
            case IGNORE_PIXEL_ASPECT :
                return 1d;
            case EXACT_PIXEL_ASPECT :
                return ratio >= 1d ? ratio : 1d;
            case MULTIPLIED_PIXEL_ASPECT :
                return ratio >= 1d ? Math.floor(ratio + 0.5d) : 1d;
            default :
                throw new InternalError("Invalid pixel aspect policy: " + pixelAspectPolicy_);
        }
    }
    /**
     * Gets the vertical pixel aspect of the image according to the
     * pixel aspect policy that is in affect.
     *
     * @return	Vertical pixel aspect.
     */
    public double getPixelAspectY() {
        if (image_ == null)
        { return 1; }
        
        Object property = image_.getProperty("aspect",this);
        if (property == null)
        { return 1; }
        double ratio = (property == null || property == Image.UndefinedProperty) ? 1d : ((Double)property).doubleValue();
        
        switch (pixelAspectPolicy_) {
            case IGNORE_PIXEL_ASPECT :
                return 1d;
            case EXACT_PIXEL_ASPECT :
                return ratio < 1d ? 1d / ratio : 1d;
            case MULTIPLIED_PIXEL_ASPECT :
                return ratio < 1d ? Math.floor(1d / ratio + 0.5d) : 1d;
            default :
                throw new InternalError("Invalid pixel aspect policy: " + pixelAspectPolicy_);
        }
    }
    /**
     * Gets the preferred image size.
     *
     * @return	Image dimension after applying the
     * pixel aspect policy.
     */
    public Dimension getPreferredImageSize() {
        if (image_ == null)
        { return new Dimension(0,0); }
        
        return new Dimension(
        (int)Math.ceil(image_.getWidth(this)*getPixelAspectX()),
        (int)Math.ceil(image_.getHeight(this)*getPixelAspectY())
        );
    }
    /**
     * Gets the scaled and pixel aspect corrected image size.
     *
     * @return	Image dimension after scaling and applying the
     * pixel aspect policy.
     */
    public Dimension getScaledImageSize() {
        if (image_ == null)
        { return new Dimension(0,0); }
        if (image_.getWidth(this) == -1) {
            try {
                MediaTracker tracker = new MediaTracker(this);
                tracker.addImage(image_,0);
                tracker.waitForID(0);
            }
            catch(InterruptedException e)
            {}
        }
        return new Dimension(
        (int)Math.ceil(image_.getWidth(this)*getPixelAspectX()*getScaleFactorX()),
        (int)Math.ceil(image_.getHeight(this)*getPixelAspectY()*getScaleFactorY())
        );
    }
    
    /**
     * Sets the image and displays it in this
     * image panel.
     *
     * @return image.
     */
    public void setImage(Image image) {
        image_ = image;
        invalidate();
        if (getParent() != null) {
            getParent().validate();
            getParent().repaint();
        }
        repaint();
    }
    /**
     * Gets the image that is displayed in this
     * image panel.
     *
     * @return image.
     */
    public Image getImage() {
        return image_;
    }
    
    /**
     * Sets the scale factor.
     * The scale factor scales images shown in
     * this image panel.
     *
     * @param	scaleX	Horizontal scale factor.
     * @param	scaleY	Vertical scale factor.
     */
    public void setScaleFactor(double scaleX, double scaleY) {
        setScaleFactor0(scaleX,scaleY);
        Component parent = getParent();
        if (parent != null) {
            parent.invalidate();
            parent.validate();
        }
        repaint();
    }
    protected void setScaleFactor0(double scaleX, double scaleY) {
        double oldScale = (scaleX_ + scaleY_) / 2d;
        scaleX_ = scaleX;
        scaleY_ = scaleY;
        propertyChangeSupport_.firePropertyChange("scale",new Double((scaleX + scaleY)/2d),new Double(oldScale));
    }
    /**
     * Gets the horizontal scale factor that is
     * used to scale images shown in this image panel.
     *
     * @return	Horizontal scale factor.
     */
    public double getScaleFactorX() {
        return scaleX_;
    }
    /**
     * Gets the vertical scale factor that is
     * used to scale images shown in this image panel.
     *
     * @return	Vertical scale factor.
     */
    public double getScaleFactorY() {
        return scaleY_;
    }
    
    /**
     * Gets the preferred size of this image panel.
     * The preferred size depends on the image size,
     * the scale factor and the pixel aspect policy.
     */
    public Dimension getPreferredSize() {
        if (image_ == null)
        { return new Dimension(50,15); }
        else {
            return getScaledImageSize();
        }
    }
    /**
     * Adds a listener who is interested in changes of this object.
     */
    public void addPropertyChangeListener(PropertyChangeListener listener) {
        propertyChangeSupport_.addPropertyChangeListener(listener);
    }
    /**
     * Removes a previously registerd listener.
     */
    public void removePropertyChangeListener(PropertyChangeListener listener) {
        propertyChangeSupport_.addPropertyChangeListener(listener);
    }
    public void setMessage(String message) {
        Vector v = new Vector();
        StringTokenizer tt = new StringTokenizer(message, "\n");
        while (tt.hasMoreTokens()) {
            v.addElement(tt.nextToken());
        }
        // Note: we assign m to the messages are only after it has
        // been completely initialized
        String[] m = new String[v.size()];
        v.copyInto(m);
        this.messages = m;
        repaint();
    }
/*
        public void setBounds(int x, int y, int w, int h)
                {
                super.setBounds(x,y,w,h);
 
                final int iw = image_.getWidth(this);
                final int ih = image_.getHeight(this);
                double xAspect = getPixelAspectX();
                double yAspect = getPixelAspectY();
 
                switch (imageScalePolicy_)
                        {
                        case SCALE_TO_FIT :
                                {
                                setScaleFactor0(iw / (double)w, ih / (double)h);
                                break;
                                }
                        case SCALE_TO_ASPECT :
                                {
                                double factor = Math.min(w / (double)(iw*xAspect), h / (double)(ih*yAspect));
                                setScaleFactor0(factor,factor);
                                break;
                                }
                        case SCALE_TO_SIZE :
                                {
                                break;
                                }
                        }
                }
 */
    
    /**
     * Workaround: Netscape gets very slow when
     * painting all SOMEBITS of an image.
     */
    public boolean imageUpdate(Image img, int flags,
    int x, int y, int w, int h) {
        if (flags == SOMEBITS) { // suppress painting of SOMEBITS.
            return true;
        }
        else {
            return super.imageUpdate(img,flags,x,y,w,h);
        }
    }
    
    
    // FIXME All the rendering hint setting code should go into a separate class.
    private  static Method setRenderingHintMethod;
    private final static Object keyInterpolation;
    private final static Object valueInterpolationNearestNeighbor;
    static {
        Method m;
        Object ki, vi;
        try {
            Class g2d = Class.forName("java.awt.Graphics2D");
            Class rh = Class.forName("java.awt.RenderingHints");
            Class rhk = Class.forName("java.awt.RenderingHints$Key");
            m = g2d.getMethod("setRenderingHint", new Class[] { rhk, Object.class });
            ki = rh.getField("KEY_INTERPOLATION").get(null);
            vi = rh.getField("VALUE_INTERPOLATION_NEAREST_NEIGHBOR").get(null);
        } catch (Exception e) {
            e.printStackTrace();
            m = null;
            ki = null;
            vi = null;
        }
        setRenderingHintMethod = m;
        keyInterpolation = ki;
        valueInterpolationNearestNeighbor = vi;
    }
    /**
     * Configures the graphics object for painting the image.
     */
    private static void configure(Graphics gr) {
        /*
        Graphics2D g = (Graphics2D) gr;
        g.setRenderingHint(
        RenderingHints.KEY_INTERPOLATION,
        RenderingHints.VALUE_INTERPOLATION_NEAREST_NEIGHBOR
        );*/
        
        if (setRenderingHintMethod != null) {
            try {
                setRenderingHintMethod.invoke(gr, new Object[] {keyInterpolation, valueInterpolationNearestNeighbor});
            } catch (Exception e) {
                e.printStackTrace();
                setRenderingHintMethod = null;
            }
        }
    }
}
