/*
 * @(#)ANIMDeltaFrame.java  1.2  2006-10-01
 *
 * Copyright (c) 1999 Werner Randelshofer
 * Staldenmattweg 2, CH-6405 Immensee, Switzerland
 * All rights reserved.
 *
 * This software is the confidential and proprietary information of
 * Werner Randelshofer. ("Confidential Information").  You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Werner Randelshofer.
 */
package ch.randelshofer.media.anim;

import java.awt.image.ColorModel;
import java.awt.image.IndexColorModel;
import java.awt.image.DirectColorModel;
import java.util.Hashtable;
import ch.randelshofer.gui.image.Bitmap;

/**
 * @author  Werner Randelshofer, Staldenmattweg 2, CH-6405 Immensee, Switzerland
 * @version 1.2 2006-10-01 Added support for DECODER_J. Removed "_" suffix from
 * instance variable names.
 * <br> 1.1 2003-03-30 Static OP codes are now public.
 * <br> 1.0  1999-10-19
 */
public class ANIMDeltaFrame
        extends ANIMFrame {
    private int leftBound,topBound,rightBound,bottomBound;
    
    private final static int
            DECODER_BYTE_VERTICAL = 5,
            DECODER_VERTICAL_7_SHORT = 6,
            DECODER_VERTICAL_7_LONG = 7,
            DECODER_VERTICAL_8_SHORT = 8,
            DECODER_VERTICAL_8_LONG = 9,
            DECODER_J = 74;
    
    public final static int
            OP_Direct = 0,
            OP_XOR = 1,
            OP_LongDelta = 2,
            OP_ShortDelta = 3,
            OP_GeneralDelta = 4,
            OP_ByteVertical = 5,
            OP_StereoDelta = 6,
            OP_Vertical7 = 7,
            OP_Vertical8 = 8,
            OP_J = 74;
    
    public ANIMDeltaFrame() {
    }
    
    private int getDecoder() {
        switch  (getOperation()) {
            case OP_Direct : // Key Frame (Data stored in ILBM BODY Chunk)
                throw new InternalError("Key Frames not yet supported (Anim Op0)");
            case OP_ByteVertical :
                if ((getBits() & BadBitsOP_ByteVertical) != 0) {
                    throw new InternalError("Unknown Bits for Anim Op5 in ANHD; Bits:" + getBits()); }
                return DECODER_BYTE_VERTICAL;
            case OP_Vertical7 :
                if ((getBits() & BIT_LongData) == 0) {
                    return DECODER_VERTICAL_7_SHORT;
                } else {
                    return DECODER_VERTICAL_7_LONG; 
                }
            case OP_Vertical8 :
                if ((getBits() & BIT_LongData) == 0) {
                    return DECODER_VERTICAL_8_SHORT; 
                } else {
                    return DECODER_VERTICAL_8_LONG;
                }
            case OP_J :
                return DECODER_J;
            default :
                throw new InternalError("ANIM Op" + getOperation() + " not supported.");
        }
    }
    
    public void decode(Bitmap bitmap, ANIMMovieTrack track) {
        switch (getDecoder()) {
            case DECODER_BYTE_VERTICAL :
                decodeByteVertical(bitmap,track);
                break;
            case DECODER_VERTICAL_7_SHORT :
                decodeVertical7Short(bitmap,track);
                break;
            case DECODER_VERTICAL_7_LONG :
                decodeVertical7Long(bitmap,track);
                break;
            case DECODER_VERTICAL_8_SHORT :
                decodeVertical8Short(bitmap,track);
                break;
            case DECODER_VERTICAL_8_LONG :
                decodeVertical8Long(bitmap,track);
                break;
            case DECODER_J :
                decodeJ(bitmap,track);
                break;
            default :
                throw new InternalError("Unsupported decoder."+getDecoder());
        }
    }
    
    private void decodeByteVertical(Bitmap bitmap, ANIMMovieTrack track) {
        int columns = 0;
        int iOp = 0;
        byte[] planeBytes = bitmap.getBitmap();
        int iPl = 0;
        int widthInBytes = bitmap.getBitplaneStride();
        int interleave = track.getNbPlanes()*widthInBytes;
        int opCode = 0;
        int opCount = 0;
        byte copyByte = 0;
        leftBound = widthInBytes;
        rightBound = 0;
        topBound = track.getHeight();
        bottomBound = 0;
        int height = track.getHeight();
        
        // Repeat for each plane.
        for (int i = 0;i < track.getNbPlanes();i++) {
            
            // iOp is the pointer (index) to the op-codes.
            iOp = ((data[i*4]&0xff)<<24)+
                    ((data[i*4+1]&0xff)<<16)+
                    ((data[i*4+2]&0xff)<<8)+
                    (data[i*4+3]&0xff);
            
            if (iOp > 0) {
                // Each column of the plane is coded on its own.
                for (columns = 0;columns < widthInBytes;columns++) {
                    
                    // Set iPl to the beginning of the column in the plane.
                    iPl = columns+i*widthInBytes;
                    opCount = data[iOp++] & 0xff;
                    
                    if (opCount > 0) {
                        if (columns < leftBound) {
                            leftBound = columns; }
                        if (columns > rightBound) {
                            rightBound = columns; }
                        opCode = data[iOp];
                        if (opCode <= 0) {
                            topBound = 0; } else {
                            if (opCode < topBound) {
                                topBound = opCode; }
                            }
                        
                        for (;opCount > 0;opCount--) {
                            opCode = data[iOp++];
                            if (opCode > 0) { // Skip ops
                                iPl+=opCode*interleave;
                            } else if (opCode < 0) { // Uniq ops
                                opCode &= 0x7f;
                                while (opCode-- > 0) {
                                    planeBytes[iPl] = data[iOp++];
                                    iPl+=interleave;
                                }
                            } else { // Repeat ops
                                opCode = data[iOp++] & 0xff;
                                if (opCode == 0) {
                                    return; } //throw new InterpretException("Error in Delta Chunk: copy bytes with count 0.");
                                copyByte = data[iOp++];
                                while (opCode-- > 0) {
                                    planeBytes[iPl] = copyByte;
                                    iPl+=interleave;
                                }
                            }
                        }
                        
                        if (opCode <= 0) {
                            int bottom = (iPl - (columns+i*widthInBytes)) / interleave;
                            if (bottom > bottomBound) {
                                bottomBound = bottom; }
                        } else {
                            if (height - opCode > bottomBound) {
                                bottomBound = height - opCode; }
                        }
                    }
                }
            }
        }
        if (leftBound <= rightBound) {
            leftBound *= 8;
            rightBound = rightBound*8 + 8;
        }
    }
    protected void decodeVertical8Short(Bitmap bitmap, ANIMMovieTrack track) {
        int columns = 0;
        int iOp = 0;
        byte[] planeBytes = bitmap.getBitmap();
        int iPl = 0;
        int widthInBytes = bitmap.getBitplaneStride();
        int interleave = track.getNbPlanes()*widthInBytes;
        int opCode = 0;
        int opCount = 0;
        byte copyByte1 = 0;
        byte copyByte2 = 0;
        leftBound = widthInBytes;
        rightBound = 0;
        topBound = track.getHeight() - 1;
        bottomBound = 0;
        int height = track.getHeight() - 1;
        
        // Repeat for each plane.
        for (int i = 0;i < track.getNbPlanes();i++) {
            
            // iOp points to the Op-Codes.
            iOp = ((data[i*4]&0xff)<<24)+
                    ((data[i*4+1]&0xff)<<16)+
                    ((data[i*4+2]&0xff)<<8)+
                    (data[i*4+3]&0xff);
            
            if (iOp > 0) {
                // Each column has its own Op-codes.
                for (columns = 0;columns < widthInBytes;columns+=2) {
                    
                    // iPl points to the column in the bitmap.
                    iPl = columns+i*widthInBytes;
                    opCount = ((data[iOp++] &0xff) << 8) | (data[iOp++] & 0xff);
                    
                    if (opCount > 0) {
                        if (columns < leftBound) {
                            leftBound = columns; }
                        if (columns > rightBound) {
                            rightBound = columns; }
                        opCode = (data[iOp] << 8) | (data[iOp+1] & 0xff);
                        if (opCode <= 0) {
                            topBound = 0; } else {
                            if (opCode < topBound) {
                                topBound = opCode; }
                            }
                        
                        for (;opCount > 0;opCount--) {
                            opCode = (data[iOp++] << 8) | (data[iOp++] & 0xff);
                            if (opCode > 0) { // Skip ops
                                iPl+=opCode*interleave;
                            } else if (opCode < 0) { // Uniq ops
                                opCode &= 0x7fff;
                                while (opCode-- > 0) {
                                    planeBytes[iPl] = data[iOp++];
                                    planeBytes[iPl+1] = data[iOp++];
                                    iPl+=interleave;
                                }
                            } else { // Repeat ops
                                opCode = ((data[iOp++] << 8) | (data[iOp++] & 0xff)) & 0xffff;
                                if (opCode == 0) {
                                    return; } //throw new InterpretException("Error in Delta Chunk: copy bytes with count 0.");
                                copyByte1 = data[iOp++];
                                copyByte2 = data[iOp++];
                                while (opCode-- > 0) {
                                    planeBytes[iPl] = copyByte1;
                                    planeBytes[iPl+1] = copyByte2;
                                    iPl+=interleave;
                                }
                            }
                        }
                        
                        if (opCode <= 0) {
                            int bottom = (iPl - (columns+i*widthInBytes)) / interleave;
                            if (bottom > bottomBound) {
                                bottomBound = bottom; }
                        } else {
                            if (height - opCode > bottomBound) {
                                bottomBound = height - opCode; }
                        }
                    }
                }
            }
        }
        if (leftBound <= rightBound) {
            leftBound *= 8;
            rightBound = rightBound*8 + 16;
        }
    }
    
    protected void decodeVertical8Long(Bitmap bitmap, ANIMMovieTrack track) {
        int columns = 0;
        int iOp = 0;
        byte[] planeBytes = bitmap.getBitmap();
        int iPl = 0;
        int widthInBytes = bitmap.getBitplaneStride();
        int interleave = track.getNbPlanes()*widthInBytes;
        int opCode = 0;
        int opCount = 0;
        byte copyByte1 = 0;
        byte copyByte2 = 0;
        byte copyByte3 = 0;
        byte copyByte4 = 0;
        leftBound = widthInBytes;
        rightBound = 0;
        topBound = track.getHeight() - 1;
        bottomBound = 0;
        int height = track.getHeight() - 1;
        
        // Repeat for each plane
        for (int i = 0;i < track.getNbPlanes();i++) {
            
            // iOp points to the op-codes.
            iOp = ((data[i*4]&0xff)<<24)+
                    ((data[i*4+1]&0xff)<<16)+
                    ((data[i*4+2]&0xff)<<8)+
                    (data[i*4+3]&0xff);
            
            if (iOp > 0) {
                // Decode each column of the plane separately.
                for (columns = 0;columns < widthInBytes;columns+=4) {
                    
                    // iPl points to the column in the bitmap.
                    iPl = columns+i*widthInBytes;
                    opCount = ((data[iOp++]&0xff)<<24)+
                            ((data[iOp++]&0xff)<<16)+
                            ((data[iOp++]&0xff)<<8)+
                            (data[iOp++]&0xff);
                    
                    if (opCount > 0) {
                        if (columns < leftBound) {
                            leftBound = columns; }
                        if (columns > rightBound) {
                            rightBound = columns; }
                        opCode = ((data[iOp]&0xff)<<24)+
                                ((data[iOp+1]&0xff)<<16)+
                                ((data[iOp+2]&0xff)<<8)+
                                (data[iOp+3]&0xff);
                        if (opCode <= 0) {
                            topBound = 0; } else {
                            if (opCode < topBound) {
                                topBound = opCode; }
                            }
                        
                        for (;opCount > 0;opCount--) {
                            opCode = ((data[iOp++]&0xff)<<24)+
                                    ((data[iOp++]&0xff)<<16)+
                                    ((data[iOp++]&0xff)<<8)+
                                    (data[iOp++]&0xff);
                            if (opCode > 0) { // Skip ops
                                iPl+=opCode*interleave;
                            } else if (opCode < 0) { // Uniq ops
                                opCode &= 0x7fffffff;
                                while (opCode-- > 0) {
                                    planeBytes[iPl] = data[iOp++];
                                    planeBytes[iPl+1] = data[iOp++];
                                    planeBytes[iPl+2] = data[iOp++];
                                    planeBytes[iPl+3] = data[iOp++];
                                    iPl+=interleave;
                                }
                            } else { // Repeat ops
                                opCode = ((data[iOp++]&0xff)<<24)+
                                        ((data[iOp++]&0xff)<<16)+
                                        ((data[iOp++]&0xff)<<8)+
                                        (data[iOp++]&0xff);
                                if (opCode == 0) {
                                    return; } //throw new InterpretException("Error in Delta Chunk: copy bytes with count 0.");
                                copyByte1 = data[iOp++];
                                copyByte2 = data[iOp++];
                                copyByte3 = data[iOp++];
                                copyByte4 = data[iOp++];
                                while (opCode-- > 0) {
                                    planeBytes[iPl] = copyByte1;
                                    planeBytes[iPl+1] = copyByte2;
                                    planeBytes[iPl+2] = copyByte3;
                                    planeBytes[iPl+3] = copyByte4;
                                    iPl+=interleave;
                                }
                            }
                        }
                        
                        if (opCode <= 0) {
                            int bottom = (iPl - (columns+i*widthInBytes)) / interleave;
                            if (bottom > bottomBound) {
                                bottomBound = bottom; }
                        } else {
                            if (height - opCode > bottomBound) {
                                bottomBound = height - opCode; }
                        }
                    }
                }
            }
        }
        if (leftBound <= rightBound) {
            leftBound *= 8;
            rightBound = rightBound*8 + 32;
        }
    }
    
    protected void decodeVertical7Short(Bitmap bitmap, ANIMMovieTrack track) {
        int columns = 0;
        int iOp = 0;
        int iData = 0;
        byte[] planeBytes = bitmap.getBitmap();
        int iPl = 0;
        int widthInBytes = bitmap.getBitplaneStride();
        int interleave = bitmap.getScanlineStride();
        int opCode = 0;
        int opCount = 0;
        byte copyByte1 = 0;
        byte copyByte2 = 0;
        leftBound = widthInBytes;
        rightBound = 0;
        topBound = track.getHeight() - 1;
        bottomBound = 0;
        int height = track.getHeight() - 1;
        
        for (int i = 0;i < track.getNbPlanes();i++) {
            iOp = ((data[i*4]&0xff)<<24)+
                    ((data[i*4+1]&0xff)<<16)+
                    ((data[i*4+2]&0xff)<<8)+
                    (data[i*4+3]&0xff);
            
            iData = ((data[i*4+32]&0xff)<<24)+
                    ((data[i*4+33]&0xff)<<16)+
                    ((data[i*4+34]&0xff)<<8)+
                    (data[i*4+35]&0xff);
            
            if (iOp > 0) {
                for (columns = 0;columns < widthInBytes;columns+=2) {
                    iPl = columns+i*widthInBytes;
                    opCount = data[iOp++] & 0xff;
                    
                    if (opCount > 0) {
                        if (columns < leftBound) {
                            leftBound = columns; }
                        if (columns > rightBound) {
                            rightBound = columns; }
                        opCode = data[iOp];
                        if (opCode <= 0) {
                            topBound = 0; } else {
                            if (opCode < topBound) {
                                topBound = opCode; }
                            }
                        
                        for (;opCount > 0;opCount--) {
                            opCode = data[iOp++];
                            if (opCode > 0) { // Skip ops
                                iPl+=opCode*interleave;
                            } else if (opCode < 0) { // Uniq ops
                                opCode &= 0x7f;
                                while (opCode-- > 0) {
                                    planeBytes[iPl] = data[iData++];
                                    planeBytes[iPl+1] = data[iData++];
                                    iPl+=interleave;
                                }
                            } else { // Repeat ops
                                opCode = data[iOp++] & 0xff;
                                if (opCode == 0) {
                                    return; } //throw new InterpretException("Error in Delta Chunk: copy bytes with count 0.");
                                copyByte1 = data[iData++];
                                copyByte2 = data[iData++];
                                while (opCode-- > 0) {
                                    planeBytes[iPl] = copyByte1;
                                    planeBytes[iPl+1] = copyByte2;
                                    iPl+=interleave;
                                }
                            }
                        }
                        
                        if (opCode <= 0) {
                            int bottom = (iPl - (columns+i*widthInBytes)) / interleave;
                            if (bottom > bottomBound) {
                                bottomBound = bottom; }
                        } else {
                            if (height - opCode > bottomBound) {
                                bottomBound = height - opCode; }
                        }
                    }
                }
            }
        }
        if (leftBound <= rightBound) {
            leftBound *= 8;
            rightBound = rightBound*8 + 32;
        }
    }
    
    protected void decodeVertical7Long(Bitmap bitmap, ANIMMovieTrack track) {
        int columns = 0;
        int iOp = 0;
        int iData = 0;
        byte[] planeBytes = bitmap.getBitmap();
        int iPl = 0;
        int widthInBytes = bitmap.getBitplaneStride();
        int interleave = track.getNbPlanes()*widthInBytes;
        int opCode = 0;
        int opCount = 0;
        byte copyByte1 = 0;
        byte copyByte2 = 0;
        byte copyByte3 = 0;
        byte copyByte4 = 0;
        leftBound = widthInBytes;
        rightBound = 0;
        topBound = track.getHeight() - 1;
        bottomBound = 0;
        int height = track.getHeight() - 1;
        
        for (int i = 0;i < track.getNbPlanes();i++) {
            iOp = ((data[i*4]&0xff)<<24)+
                    ((data[i*4+1]&0xff)<<16)+
                    ((data[i*4+2]&0xff)<<8)+
                    (data[i*4+3]&0xff);
            
            iData = ((data[i*4+32]&0xff)<<24)+
                    ((data[i*4+33]&0xff)<<16)+
                    ((data[i*4+34]&0xff)<<8)+
                    (data[i*4+35]&0xff);
            
            if (iOp > 0) {
                for (columns = 0;columns < widthInBytes;columns+=4) {
                    iPl = columns+i*widthInBytes;
                    opCount = data[iOp++] & 0xff;
                    
                    if (opCount > 0) {
                        if (columns < leftBound) {
                            leftBound = columns; }
                        if (columns > rightBound) {
                            rightBound = columns; }
                        opCode = data[iOp];
                        if (opCode <= 0) {
                            topBound = 0; } else {
                            if (opCode < topBound) {
                                topBound = opCode; }
                            }
                        
                        for (;opCount > 0;opCount--) {
                            opCode = data[iOp++];
                            if (opCode > 0) { // Skip ops
                                iPl+=opCode*interleave;
                            } else if (opCode < 0) { // Uniq ops
                                opCode &= 0x7f;
                                while (opCode-- > 0) {
                                    planeBytes[iPl] = data[iData++];
                                    planeBytes[iPl+1] = data[iData++];
                                    planeBytes[iPl+2] = data[iData++];
                                    planeBytes[iPl+3] = data[iData++];
                                    iPl+=interleave;
                                }
                            } else { // Repeat ops
                                opCode = data[iOp++] & 0xff;
                                if (opCode == 0) {
                                    return; } //throw new InterpretException("Error in Delta Chunk: copy bytes with count 0.");
                                copyByte1 = data[iData++];
                                copyByte2 = data[iData++];
                                copyByte3 = data[iData++];
                                copyByte4 = data[iData++];
                                while (opCode-- > 0) {
                                    planeBytes[iPl] = copyByte1;
                                    planeBytes[iPl+1] = copyByte2;
                                    planeBytes[iPl+2] = copyByte3;
                                    planeBytes[iPl+3] = copyByte4;
                                    iPl+=interleave;
                                }
                            }
                        }
                        
                        if (opCode <= 0) {
                            int bottom = (iPl - (columns+i*widthInBytes)) / interleave;
                            if (bottom > bottomBound) {
                                bottomBound = bottom; }
                        } else {
                            if (height - opCode > bottomBound) {
                                bottomBound = height - opCode; }
                        }
                    }
                }
            }
        }
        if (leftBound <= rightBound) {
            leftBound *= 8;
            rightBound = rightBound*8 + 64;
        }
    }
    
    /** Decodes DLTA's in Eric Graham's Compresson mode "J".
     *
     * The following documentation has been taken from Steven Den Beste's
     * docu for his "unmovie" program.
     *
     * A DLTA appears to have three kinds
     * of items in it, with each type being indicated by the value of its first byte:
     *
     *
     * Type 0: indicates the end of the DLTA. Layout:
     * word: 0
     *
     * Type 1: indicates a "wall": This is a section of the image which has full
     * Z-height, is 1 byte wide in X, and has a variable Y size. Layout:
     * word: 1
     * word: 0=unidirectional (store value), 1=bidirectional (XOR value)
     * word: Y-size (number of pixels in Y direction)
     * word: number of blocks to follow:
     * per block:
     * word: offset in each bitplane (note: NOT in the total image!)
     * 1-6 bytes: full Z height for first Y
     * 1-6 bytes: full Z height for second Y
     * etc., extending DOWN.
     *
     * Type 2: indicates a "pile": This is a section of the image which has full
     * Z-height, and has both variable Y size and X size. Layout:
     * word: 2
     * word: 0=unidirectional, 1=bidirectional
     * word: Y size
     * word: X size
     * word: number of blocks to follow:
     * per block:
     * word: offset in each bitplane (NOT in the total image)
     * successive bytes: a traversed 3D rectangle, with X varying within
     * Y within Z. (X moves right, Z moves up, Y moves down)
     *
     * The movie is double-buffered, but you don't have to know about that part.
     * (Anyway, it is described in the original documentation for "pilbm" if you're
     * curious.
     */
    protected void decodeJ(Bitmap bitmap, ANIMMovieTrack track) {
        
        int nbPlanes = track.getNbPlanes();
        int widthInBytes = bitmap.getBitplaneStride();
        
        // Mark all pixels of the delta frame as being changed
        // XXX - Determine minimal bounds
        /*
        rightBound = track.getWidth();
        leftBound = 0;
        bottomBound = track.getHeight();
        topBound = 0;*/
        leftBound = track.getWidth() - 1;
        rightBound = 0;
        topBound = track.getHeight() - 1;
        bottomBound = 0;
        
        // Current reading position;
        int pos = 0;
        
        // Output data goes here:
        byte[] planeBytes = bitmap.getBitmap();
        
        // Change type, 16 bit short: 0=End of Delta, 1=Wall, 2=Pile.
        int changeType;
        
        decodingLoop: while (pos < data.length) {
            changeType = ((data[pos++] & 0xff) << 8) | ((data[pos++]) & 0xff);
            
            switch(changeType) {
                case 0: /* End of DELTA */
                    break decodingLoop;
                    
                case 1: { /* Wall */
                    // Read wall header
                    // struct {
                    //        short uni_flag;
                    //      short y_size;
                    //      short num_blocks; } wall;
                    int uniFlag = ((data[pos++] & 0xff) << 8) | ((data[pos++]) & 0xff);
                    int ySize = ((data[pos++] & 0xff) << 8) | ((data[pos++]) & 0xff);
                    int numBlocks = ((data[pos++] & 0xff) << 8) | ((data[pos++]) & 0xff);
                    
                    // Decode wall data
                    for (int b=0; b < numBlocks; b++) {
                        int offset = ((data[pos++] & 0xff) << 8) | ((data[pos++]) & 0xff);
                        
                        leftBound = Math.min(leftBound, (offset % widthInBytes) * 8);
                        rightBound = Math.max(rightBound, (offset % widthInBytes) * 8 + 8);
                        topBound = Math.min(topBound, (offset / widthInBytes));
                        bottomBound = Math.max(bottomBound, (offset / widthInBytes) + ySize);
                        
                        int realOffset = (offset / widthInBytes) * nbPlanes;
                        realOffset *= widthInBytes;
                        realOffset += offset % widthInBytes;
                        
                        for (int z=0; z < nbPlanes; z++) {
                            for (int y=0; y < ySize; y++) {
                                byte charval = data[pos++];
                                int dest = z * widthInBytes * ySize +
                                        y * widthInBytes +
                                        realOffset;
                                if (uniFlag == 1) {
                                    planeBytes[dest] ^= charval;
                                } else {
                                    planeBytes[dest] = charval;
                                }
                            }
                        }
                        
                        // If we've stopped on an odd boundary, read and throw away
                        // another byte.
                        if (pos % 2 == 1) pos++;
                    }
                    break;
                }
                case 2: { /* Pile */
                    // Read Pile header
                    // struct {
                    //    short uni_flag;
                    //    short y_size;
                    //    short x_size;
                    //    short num_blocks; } pile;
                    int uniFlag = ((data[pos++] & 0xff) << 8) | ((data[pos++]) & 0xff);
                    int ySize = ((data[pos++] & 0xff) << 8) | ((data[pos++]) & 0xff);
                    int xSize = ((data[pos++] & 0xff) << 8) | ((data[pos++]) & 0xff);
                    int numBlocks = ((data[pos++] & 0xff) << 8) | ((data[pos++]) & 0xff);
                    
                    // Decode Pile data
                    for (int b=0; b < numBlocks; b++) {
                        int offset = ((data[pos++] & 0xff) << 8) | ((data[pos++]) & 0xff);
                        
                        leftBound = Math.min(leftBound, (offset % widthInBytes) * 8);
                        rightBound = Math.max(rightBound, (offset % widthInBytes + xSize) * 8 + 8);
                        topBound = Math.min(topBound, (offset / widthInBytes));
                        bottomBound = Math.max(bottomBound, (offset / widthInBytes) + ySize);
                        
                        int realOffset = (offset / widthInBytes) * nbPlanes;
                        realOffset *= widthInBytes;
                        realOffset += offset % widthInBytes;
                        
                        for (int z=0; z < nbPlanes; z++) {
                            for (int y=0; y < ySize; y++) {
                                for (int x=0; x < xSize; x++) {
                                    byte charval = data[pos++];
                                    int dest = z * widthInBytes * ySize +
                                            y * widthInBytes +
                                            realOffset + x;
                                    if (uniFlag == 1) {
                                        planeBytes[dest] ^= charval;
                                    } else if (uniFlag == 0) {
                                        planeBytes[dest] = charval;
                                    }
                               // planeBytes[dest] = (byte) 0x55;
                                }
                            }
                        }
                        
                        // If we've stopped on an odd boundary, read and throw away
                        // another byte.
                        if (pos % 2 == 1) pos++;
                    }
                    break;
                }
                default:
                    System.out.println("Unsupported changeType in 'J' delta frame:"+changeType);
                    break decodingLoop;
                    //throw new InternalError("Unsupported changeType in 'J' delta frame:"+changeType);
            }
        }
        //topBound *= 2;
        
        //rightBound = track.getWidth() - 1;
        //leftBound = 0;
        //bottomBound = track.getHeight() - 1;
        
        /*
delta_proc(buffer)
char *buffer;
{
    short	change_type;
    long	seek_val;
    int	b;
    short	offset;
    long	x, y, z;
    char	charval;
    long	realOffset;
    char	*dest;
         
    while(!feof(in)) {
        fread(&change_type, (size_t)sizeof(short), (size_t)1, in);
         
        switch(change_type) {
        case 0: /* End of DELTA * /
            return;
         
        case 1: /* Wall * /
            fread(&wall, (size_t)sizeof(wall), (size_t)1, in);
            for (b=0; b<wall.num_blocks; b++) {
                fread(&offset, (size_t)2, (size_t)1, in);
         
                realOffset = ((long)offset/widthInBytes) * ilbmbmhd.nPlanes;
                realOffset *= widthInBytes;
                realOffset += offset % widthInBytes;
         
                for (z=0; z<ilbmbmhd.nPlanes; z++) {
                    for (y=0; y<wall.y_size; y++) {
                        fread(&charval, (size_t)1, (size_t)1, in);
                        dest = buffer;
                        dest += z * widthInBytes*wall.y_size;
                        dest += y * widthInBytes;
                        dest += realOffset;
                        if (wall.uni_flag == 1)
         *dest ^= charval;
                        else
         *dest = charval;
                    }
                }
         
                /*
         * If we've stopped on an odd boundary, read and throw away
         * another byte.
         * /
                seek_val = ftell(in);
                if (seek_val & 0x01L)
                    fread(&charval, (size_t)1, (size_t)1, in);
            }
            break;
         
        case 2: /* Pile * /
            fread(&pile, (size_t)sizeof(pile), (size_t)1, in);
            for (b=0; b<pile.num_blocks; b++) {
                fread(&offset, (size_t)2, (size_t)1, in);
                realOffset = ((long)offset/widthInBytes) * ilbmbmhd.nPlanes;
                realOffset *= widthInBytes;
                realOffset += offset % widthInBytes;
         
                for (z=0; z<ilbmbmhd.nPlanes; z++) {
                    for (y=0; y<pile.y_size; y++) {
                        for (x=0; x<pile.x_size; x++) {
                            fread(&charval, (size_t)1, (size_t)1, in);
                            dest = buffer;
                            dest += z * widthInBytes*pile.y_size;
                            dest += y * widthInBytes;
                            dest += realOffset + x;
                            if (pile.uni_flag == 1)
         *dest ^= charval;
                            else
         *dest = charval;
                        }
                    }
                }
         
                /*
         * If we've stopped on an odd boundary, read and throw away
         * another byte.
         * /
                seek_val = ftell(in);
                if (seek_val & 0x01L)
                    fread(&charval, (size_t)1, (size_t)1, in);
            }
            break;
         
        default:
            printf("I seem to have found change_type %d\n", change_type);
            printf("I didn't think there was such a thing!\n");
            exit(1);
            break;
        }
    }
}
         */
    }
    
    
    public int getTopBound(ANIMMovieTrack track) { return topBound; }
    public int getBottomBound(ANIMMovieTrack track) { return bottomBound; }
    public int getLeftBound(ANIMMovieTrack track) { return leftBound; }
    public int getRightBound(ANIMMovieTrack track) { return rightBound; }
}
