/*
 * @(#)AppletUtil.java  2.0  2004-12-27
 *
 * Copyright (c) 2000 Werner Randelshofer
 * Staldenmattweg 2, CH-6405 Immensee, Switzerland
 * All rights reserved.
 *
 * This software is the confidential and proprietary information of
 * Werner Randelshofer. ("Confidential Information").  You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Werner Randelshofer.
 */
package ch.randelshofer.util;

import java.applet.Applet;
import java.awt.*;
import java.util.*;

/**
 * Usefull methods for applets.
 *
 * @author Werner Randelshofer
 * @version 2.0 2004-12-27 Enhanced to support the special applet parameters
 * of Sun's Java plug in. See http://java.sun.com/j2se/1.5.0/docs/guide/plugin/developer_guide/special_attributes.html
 * <br>1.3 2003-07-25 Method
 * getIndexedKeyValueParameters(Applet,String,Hashtable) added.
 * <br>1.2 2002-06-26 Method getParameter(Applet,String,double) added.
 * <br>1.1 2001-12-27 Method getParameters did not work.
 * <br>1.0 2000-03-06
 */
public class Applets extends Object {
    /**
     * HTML 4.1 standard color names.
     * @see http://www.w3.org/TR/html4/types.html#h-6.5
     */
    private final static Hashtable htmlColors = new Hashtable();
    static {
        htmlColors.put("black", new Color(0x000000));
        htmlColors.put("green", new Color(0x008000));
        htmlColors.put("silver", new Color(0xC0C0C0));
        htmlColors.put("lime", new Color(0x00FF00));
        htmlColors.put("gray", new Color(0x808080));
        htmlColors.put("olive", new Color(0x808000));
        htmlColors.put("white", new Color(0xFFFFFF));
        htmlColors.put("yellow", new Color(0xFFFF00));
        htmlColors.put("maroon", new Color(0x800000));
        htmlColors.put("navy", new Color(0x000080));
        htmlColors.put("red", new Color(0xFF0000));
        htmlColors.put("blue", new Color(0x0000FF));
        htmlColors.put("purple", new Color(0x800080));
        htmlColors.put("teal", new Color(0x008080));
        htmlColors.put("fuchsia", new Color(0xFF00FF));
        htmlColors.put("aqua", new Color(0x00FFFF));
    }
    /**
     * Java standard color names.
     * @see java.awt.Color
     */
    private final static Hashtable javaColors = new Hashtable();
    static {
        javaColors.put("white", Color.white);
        javaColors.put("lightgray", Color.lightGray);
        javaColors.put("gray", Color.gray);
        javaColors.put("darkgray", Color.darkGray);
        javaColors.put("black", Color.black);
        javaColors.put("red", Color.red);
        javaColors.put("pink", Color.pink);
        javaColors.put("orange", Color.orange);
        javaColors.put("yellow", Color.yellow);
        javaColors.put("green", Color.green);
        javaColors.put("magenta", Color.magenta);
        javaColors.put("cyan", Color.cyan);
        javaColors.put("blue", Color.blue);
    }
    
    /**
     * Reads a parameter as a color value.
     * <p>
     * The color can be represented in the following formats:
     * <ul>
     * <li>any Color from java.awt.Color;
     *
     * r,g,b where r, g, and b are integers in the range of 0-255 that would render an opaque standard RGB (sRGB) color in the Color constructor Color(int r, int g, int b);
     *
     * standard HTML colors: silver, green, maroon, purple, navy, teal, and olive; or
     *
     * hexadecimal color forma</li>
     * <li></li>
     * <li></li>
     * <li></li>
     * </ul>
     *
     * @param applet The applet.
     * @param name   The name of the parameter.
     * @param defaultValue The default value.
     *
     * @return Returns the value of the parameter, or the default value, if
     * the parameter does not exist, or does not represent a double.
     */
    public static Color getParameter(Applet applet, String name, Color defaultValue) {
        String value = applet.getParameter(name);
        if (value != null) {
            value = value.toLowerCase();
            if (htmlColors.contains(value)) {
                return (Color) htmlColors.get(value);
            }
            if (javaColors.contains(value)) {
                return (Color) javaColors.get(value);
            }
            int[] ints = getParameters(applet, name, new int[0]);
            switch (ints.length ) {
                case 1 :
                    return new Color(ints[0]);
                case 3 :
                    return new Color(ints[0], ints[1], ints[2]);
            }
        }
        return defaultValue;
    }
    
    /**
     * Suppresses default constructor, ensuring non-instantiability.
     */
    private Applets() {
    }
    
    /**
     * Returns the String value of an applet parameter or the
     * default value if the parameter is null.
     *
     * @param   applet  The applet from which we want to get the parameter.
     * @param   name    The name of the parameter.
     * @param   defaultValue This value is returned when the applet returns
     *                  null for the parameter.
     */
    public static String getParameter(Applet applet, String name, String defaultValue) {
        String value = applet.getParameter(name);
        return value != null ? value : defaultValue;
    }
    
    /**
     * Returns the parameter value of the specified applet or the
     * default value if the parameter is null, the value is
     * returned as an array. The values of the array consist
     * of each word found in the value. The words are delimited
     * by spaces or by commas.
     *
     * @param   applet  The applet from which we want to get the parameter.
     * @param   name    The name of the parameter.
     * @param   defaultValue This value is returned when the applet returns
     *                  null for the parameter.
     */
    public static String[] getParameters(Applet applet, String name, String[] defaultValue) {
        String param = applet.getParameter(name);
        if (param != null) {
            StringTokenizer scanner = new StringTokenizer(param, ", ");
            String[] values = new String[scanner.countTokens()];
            for (int i = 0; i < values.length; i++) {
                values[i] = scanner.nextToken();
            }
            return values;
        }
        return defaultValue;
    }
    
    /**
     * Reads a parameter as an int value.
     *
     * @param applet The applet.
     * @param name   The name of the parameter.
     * @param defaultValue The default value.
     *
     * @return Returns the value of the parameter, or the default value, if
     * the parameter does not exist, or does not represent an int.
     */
    public static int getParameter(Applet applet, String name, int defaultValue) {
        String value = applet.getParameter(name);
        if (value != null) {
            try {
                //return Integer.parseInt(value);
                return decode(value).intValue();
            } catch (NumberFormatException e) {}
        }
        
        return defaultValue;
    }
    
    /**
     * Reads a parameter as a boolean value.
     *
     * @param applet The applet.
     * @param name   The name of the parameter.
     * @param defaultValue The default value.
     *
     * @return Returns the value of the parameter, or the default value, if
     * the parameter does not exist, or does not represent a boolean.
     */
    public static boolean getParameter(Applet applet, String name, boolean defaultValue) {
        String value = applet.getParameter(name);
        if (value != null) {
            value = value.toLowerCase();
            if (value.equals("true")) {
                return true;
            } else if (value.equals("false")) {
                return false;
            }
        }
        return defaultValue;
    }
    
    /**
     * Reads a parameter as an array of comma separated int values.
     *
     * @param applet The applet.
     * @param name   The name of the parameter.
     * @param defaultValue The default value.
     *
     * @return Returns the value of the parameter, or the default value, if
     * the parameter does not exist, or does not represent an array of int's.
     */
    public static int[] getParameters(Applet applet, String name, int[] defaultValue) {
        String param = applet.getParameter(name);
        if (param != null) {
            try {
                StringTokenizer scanner = new StringTokenizer(param, ", ");
                int[] values = new int[scanner.countTokens()];
                for (int i = 0; i < values.length; i++) {
                    //values[i] = Integer.parseInt(scanner.nextToken());
                    values[i] = decode(scanner.nextToken()).intValue();
                }
                return values;
            } catch (NumberFormatException e) {}
        }
        return defaultValue;
    }
    
    /**
     * Reads a parameter as a double value.
     *
     * @param applet The applet.
     * @param name   The name of the parameter.
     * @param defaultValue The default value.
     *
     * @return Returns the value of the parameter, or the default value, if
     * the parameter does not exist, or does not represent a double.
     */
    public static double getParameter(Applet applet, String name, double defaultValue) {
        String value = applet.getParameter(name);
        if (value != null) {
            try {
                //return Integer.parseInt(value);
                return Double.valueOf(value).doubleValue();
            } catch (NumberFormatException e) {}
        }
        
        return defaultValue;
    }
    
    /**
     * Returns the parameter value of the specified applet or the
     * default value if the parameter is null, the value is
     * returned as a Hashtable.
     * <p>
     * The parameter value is parsed according to the following EBNF production:
     * <pre>
     * list ::= entry {"," entry}
     * entry ::= [key "="] value
     * key ::= String
     * value ::= String
     * </pre>
     * <p>
     * The returned Hashtable contains up to two key value pairs for each entry.
     * <br>If an entry does specify a key, it is put using the specified key into
     * the hashtable.
     * <br>For each entry a key is generated which denotes the index of the entry.
     * If the Hashtable does not contain an entry for the generated key. Then an
     * entry is put using the generated key as well.
     *
     * @param   applet  The applet from which we want to get the parameter.
     * @param   name    The name of the parameter.
     * @param   defaultValue This value is returned when the applet returns
     *                  null for the parameter.
     * @return  A hashtable containing a key value pair for each entry.
     */
    public static Hashtable getIndexedKeyValueParameters(Applet applet, String name, Hashtable defaultValue) {
        String entry, key, value, generatedKey;
        int pos;
        String param = applet.getParameter(name);
        if (param != null) {
            Hashtable map = new Hashtable();
            StringTokenizer scanner = new StringTokenizer(param, ", ");
            int count = scanner.countTokens();
            for (int i = 0; i < count; i++) {
                entry = scanner.nextToken();
                pos = entry.indexOf('=');
                if (pos < 1) {
                    key = null;
                    value = entry;
                } else {
                    key = entry.substring(0, pos);
                    value = entry.substring(pos + 1);
                }
                generatedKey = Integer.toString(i);
                if (key != null) {
                    map.put(key, value);
                }
                if (! map.contains(generatedKey)) {
                    map.put(generatedKey, value);
                }
            }
            return map;
        }
        return defaultValue;
    }
    
    /**
     * Decodes a <code>String</code> into an <code>Integer</code>.  Accepts
     * decimal, hexadecimal, and octal numbers, in the following formats:
     * <pre>
     *     [-]    <decimal constant>
     *     [-] 0x     <hex constant>
     *     [-] #      <hex constant>
     *     [-] 0    <octal constant>
     * </pre>
     *
     * The constant following an (optional) negative sign and/or "radix
     * specifier" is parsed as by the <code>Integer.parseInt</code> method
     * with the specified radix (10, 8 or 16).  This constant must be positive
     * or a NumberFormatException will result.  The result is made negative if
     * first character of the specified <code>String</code> is the negative
     * sign.  No whitespace characters are permitted in the
     * <code>String</code>.
     *
     * @param     nm the <code>String</code> to decode.
     * @return    the <code>Integer</code> represented by the specified string.
     * @exception NumberFormatException  if the <code>String</code> does not
     *            contain a parsable integer.
     * @see java.lang.Integer#parseInt(String, int)
     */
    public static Integer decode(String nm) throws NumberFormatException {
        int radix = 10;
        int index = 0;
        boolean negative = false;
        Integer result;
        
        // Handle minus sign, if present
        if (nm.startsWith("-")) {
            negative = true;
            index++;
        }
        
        // Handle radix specifier, if present
        if (nm.startsWith("0x", index) || nm.startsWith("0X", index)) {
            index += 2;
            radix = 16;
        } else if (nm.startsWith("#", index)) {
            index ++;
            radix = 16;
        } else if (nm.startsWith("0", index) && nm.length() > 1 + index) {
            index ++;
            radix = 8;
        }
        
        if (nm.startsWith("-", index))
            throw new NumberFormatException("Negative sign in wrong position");
        
        try {
            result = Integer.valueOf(nm.substring(index), radix);
            result = negative ? new Integer(-result.intValue()) : result;
        } catch (NumberFormatException e) {
            // If number is Integer.MIN_VALUE, we'll end up here. The next line
            // handles this case, and causes any genuine format error to be
            // rethrown.
            String constant = negative ? new String("-" + nm.substring(index))
            : nm.substring(index);
            result = Integer.valueOf(constant, radix);
        }
        return result;
    }
}
