/*
 * @(#)StateTracker.java  1.0  1999-10-19
 *
 * Copyright (c) 1999 Werner Randelshofer
 * Staldenmattweg 2, CH-6405 Immensee, Switzerland
 * All rights reserved.
 *
 * This software is the confidential and proprietary information of
 * Werner Randelshofer. ("Confidential Information").  You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Werner Randelshofer.
 */
package ch.randelshofer.util;

/**
 * Tracks state changes in a StateModel.
 *
 * @author  Werner Randelshofer, Staldenmattweg 2, CH-6405 Immensee, Switzerland
 * @version    1.0  1999-10-19
 */
public class StateTracker
implements StateListener {
    
    private StateModel model_;
    private int[] targetStates_;
    private int targetStateIndex_;
    
    /**
     * Creates a StateTracker for the indicated StateModel.
     *
     * @param  StateModel to be tracked.
     */
    public StateTracker(StateModel model) {
        setStateModel(model);
    }
    
    /**
     * Sets the StateModel.
     * Note: This method must not be called while one of the
     *       waitForState methods is working.
     * @param  StateModel to be tracked.
     */
    public void setStateModel(StateModel model) {
        if (model_ != null) {
            model_.removeStateListener(this);
        }
        
        model_ = model;
        
        if (model_ != null) {
            model_.addStateListener(this);
        }
    }
    
    /**
     * Waits until the StateModel reaches the indicated
     * state.
     * Note: waitForState methods may not be called from
     *       multiple threads simoultaneously.
     *
     * @param  state  to wait for.
     */
    public void waitForState(int state) {
        int[] statelist = { state };
        waitForState( statelist );
    }
    
    /**
     * Waits until the StateModel reaches one of the indicated
     * states.
     *
     * Note: waitForState methods may not be called from
     *       multiple threads simoultaneously.
     *
     * @param  states  choice of states to wait for.
     */
    public int waitForState(int[] states) {
        synchronized (this) {
            targetStates_ = states;
            targetStateIndex_ = -1;
            int state = model_.getState();
            for (int i=0; i < targetStates_.length; i++) {
                if (state == targetStates_[i]) {
                    return targetStates_[i];
                }
            }
            
            while (targetStateIndex_ == -1) {
                try { wait(); } catch (InterruptedException e) {}
            }
        }
        
        return targetStates_[targetStateIndex_];
    }
    
    /**
     * XXX This method is public as an implementation side effect.
   " Do not call or override.
     */
    public void stateChanged(StateEvent event) {
        synchronized (this) {
            if (targetStates_ != null
            && targetStateIndex_ == -1) {
                int state = event.getNewState();
                
                for (int i=0; i < targetStates_.length; i++) {
                    if (state == targetStates_[i]) {
                        targetStateIndex_ = i;
                        notifyAll();
                        break;
                    }
                }
            }
        }
    }
}
