/*
 * @(#)AudioFormat.java  1.0  2011-07-10
 * 
 * Copyright (c) 2011 Werner Randelshofer, Immensee, Switzerland.
 * All rights reserved.
 * 
 * You may not use, copy or modify this file, except in compliance with the
 * license agreement you entered into with Werner Randelshofer.
 * For details see accompanying license terms.
 */
package ch.randelshofer.media;

import java.nio.ByteOrder;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import javax.sound.sampled.AudioFormat.Encoding;

/**
 * {@code AudioFormat}.
 *
 * @author Werner Randelshofer
 * @version 1.0 2011-07-10 Created.
 */
public class AudioFormat implements Format {

    /**
     * Specifies signed, linear PCM data.
     */
    public static final String PCM_SIGNED = javax.sound.sampled.AudioFormat.Encoding.PCM_SIGNED.toString();
    /**
     * Specifies unsigned, linear PCM data.
     */
    public static final String PCM_UNSIGNED = javax.sound.sampled.AudioFormat.Encoding.PCM_UNSIGNED.toString();
    /**
     * Specifies u-law encoded data.
     */
    public static final String ULAW = javax.sound.sampled.AudioFormat.Encoding.ULAW.toString();
    /**
     * Specifies a-law encoded data.
     */
    public static final String ALAW = javax.sound.sampled.AudioFormat.Encoding.ALAW.toString();
    /**
     * Specifies MP3 encoded data.
     */
    public static final String MP3 = "MP3";
    
    private final String encoding;
    private final float sampleRate;
    private final int sampleSizeInBits;
    private final int channels;
    private final int frameSize;
    private final float frameRate;
    private final ByteOrder byteOrder;
    private HashMap<String,Object>properties;

    public AudioFormat(String encoding) {
        this(encoding,NOT_SPECIFIED,NOT_SPECIFIED,NOT_SPECIFIED,NOT_SPECIFIED,NOT_SPECIFIED,null);
    }
    public AudioFormat(String encoding, ByteOrder byteOrder) {
        this(encoding,NOT_SPECIFIED,NOT_SPECIFIED,NOT_SPECIFIED,NOT_SPECIFIED,NOT_SPECIFIED,byteOrder);
    }

    
    
    public AudioFormat(String encoding, float sampleRate, int sampleSizeInBits,
		       int channels, int frameSize, float frameRate, ByteOrder byteOrder) {
        this.encoding=encoding;
        this.sampleRate=sampleRate;
        this.sampleSizeInBits=sampleSizeInBits;
        this.channels=channels;
        this.frameSize=frameSize;
        this.frameRate=frameRate;
        this.byteOrder=byteOrder;
    }
    public AudioFormat(float sampleRate, int sampleSizeInBits,
		       int channels, boolean signed, boolean bigEndian) {

	this((signed == true ? PCM_SIGNED : PCM_UNSIGNED),
	     sampleRate,
	     sampleSizeInBits,
	     channels,
	     (channels == NOT_SPECIFIED || sampleSizeInBits == NOT_SPECIFIED)?
	     NOT_SPECIFIED:
	     ((sampleSizeInBits + 7) / 8) * channels,
	     sampleRate,
	     bigEndian?ByteOrder.BIG_ENDIAN:ByteOrder.LITTLE_ENDIAN);
    }

    
    public AudioFormat(javax.sound.sampled.AudioFormat fmt) {
         this.encoding=fmt.getEncoding().toString();
        this.sampleRate=fmt.getSampleRate();
        this.sampleSizeInBits=fmt.getSampleSizeInBits();
        this.channels=fmt.getChannels();
        this.frameSize=fmt.getFrameSize();
        this.frameRate=fmt.getFrameRate();
        this.byteOrder=fmt.isBigEndian()?ByteOrder.BIG_ENDIAN:ByteOrder.LITTLE_ENDIAN;
        this.properties=new HashMap(fmt.properties());
    }

    /** The type of encoding for sounds in this format. */
    @Override
    public String getEncoding() {
        return encoding;
    }

    /** The byte order of the audio data. */
    public ByteOrder getByteOrder() {
        return byteOrder;
    }

    /**
     * The size of a sample. For compressed formats, the return value is 
     * the sample size of the uncompressed audio data. 
     */
    public int getSampleSizeInBits() {
        return sampleSizeInBits;
    }

    /** The sample rate. For compressed formats, the return value is the
     * sample rate of the uncompressed audio data. 
     */
    public float getSampleRate() {
        return sampleRate;
    }

    /** The number of channels. */
    public int getChannels() {
        return channels;
    }

    /** The frame size in bytes. */
    public int getFrameSize() {
        return frameSize;
    }

    /** The frame rate in frames per second. */
    public float getFrameRate() {
        return frameRate;
    }

    /** The {@code javax.sound.sampled.AudioFormat} representation of this
     * audio format.
     */ 
    public javax.sound.sampled.AudioFormat getJavaxSoundAudioFormat() {
return new javax.sound.sampled.AudioFormat(new Encoding(encoding),
                sampleRate, sampleSizeInBits, channels, frameSize, frameRate, byteOrder==ByteOrder.BIG_ENDIAN,properties);    

    }
    
    public Map<String,Object> getProperties() {
        return (properties==null)?Collections.EMPTY_MAP:Collections.unmodifiableMap(properties);
    }

    @Override
    public String toString() {
        return "AudioFormat{" /*+ "encoding="*/ + encoding + ", sampleRate=" + sampleRate + ", sampleSizeInBits=" + sampleSizeInBits + ", channels=" + channels + ", frameSize=" + frameSize + ", frameRate=" + frameRate + ", byteOrder=" + byteOrder + '}';
    }

    
    
    @Override
    public boolean matches(Format fmt) {
        if (fmt instanceof AudioFormat) {
            AudioFormat that = (AudioFormat) fmt;
            return (this.encoding==null||that.encoding==null||this.encoding.equals(that.encoding))
                    //  && (this.dataClass == null || that.dataClass == null || this.dataClass.equals(that.dataClass))
                    && (this.sampleSizeInBits == NOT_SPECIFIED || that.sampleSizeInBits == NOT_SPECIFIED || this.sampleSizeInBits == that.sampleSizeInBits)
                    && (this.sampleRate == NOT_SPECIFIED || that.sampleRate == NOT_SPECIFIED || this.sampleRate == that.sampleRate)
                    && (this.frameRate == NOT_SPECIFIED || that.frameRate == NOT_SPECIFIED || this.frameRate == that.frameRate)
                    && (this.frameSize == NOT_SPECIFIED || that.frameSize == NOT_SPECIFIED || this.frameSize == that.frameSize)
                    && (this.channels == NOT_SPECIFIED || that.channels == NOT_SPECIFIED || this.channels == that.channels)
                    && (this.byteOrder == null || that.byteOrder == null || this.byteOrder == that.byteOrder);
        }
        return false;
    }
    
}
