/*
 * @(#)BitmapCodec.java  1.0  2011-02-20
 * 
 * Copyright (c) 2011 Werner Randelshofer, Immensee, Switzerland.
 * All rights reserved.
 * 
 * You may not use, copy or modify this file, except in compliance with the
 * license agreement you entered into with Werner Randelshofer.
 * For details see accompanying license terms.
 */
package ch.randelshofer.media.image;

import ch.randelshofer.media.AbstractVideoCodec;
import ch.randelshofer.media.Buffer;
import ch.randelshofer.media.Codec;
import ch.randelshofer.media.Format;
import ch.randelshofer.media.VideoFormat;
import ch.randelshofer.media.ilbm.ColorCyclingMemoryImageSource;
import ch.randelshofer.media.ilbm.ILBMDecoder;
import ch.randelshofer.media.pbm.PBMDecoder;
import java.awt.image.BufferedImage;
import java.io.BufferedInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import javax.imageio.ImageIO;

/**
 * Decodes media data into a {@code Bitmap}.
 *
 * @author Werner Randelshofer
 * @version 1.0 2011-02-20 Created.
 */
public class BitmapCodec extends AbstractVideoCodec {
    public BitmapCodec() {
        super(new Format[]{
                    new VideoFormat(VideoFormat.IMAGE), //
                },
                new Format[]{
                    new VideoFormat(VideoFormat.IFF_INTERLEAVED_BITMAP, byte[].class, Format.NOT_SPECIFIED, Format.NOT_SPECIFIED, 1),//
                    new VideoFormat(VideoFormat.IFF_INTERLEAVED_BITMAP, byte[].class, Format.NOT_SPECIFIED, Format.NOT_SPECIFIED, 2),//
                    new VideoFormat(VideoFormat.IFF_INTERLEAVED_BITMAP, byte[].class, Format.NOT_SPECIFIED, Format.NOT_SPECIFIED, 3),//
                    new VideoFormat(VideoFormat.IFF_INTERLEAVED_BITMAP, byte[].class, Format.NOT_SPECIFIED, Format.NOT_SPECIFIED, 4),//
                    new VideoFormat(VideoFormat.IFF_INTERLEAVED_BITMAP, byte[].class, Format.NOT_SPECIFIED, Format.NOT_SPECIFIED, 5),//
                    new VideoFormat(VideoFormat.IFF_INTERLEAVED_BITMAP, byte[].class, Format.NOT_SPECIFIED, Format.NOT_SPECIFIED, 6),//
                    new VideoFormat(VideoFormat.IFF_INTERLEAVED_BITMAP, byte[].class, Format.NOT_SPECIFIED, Format.NOT_SPECIFIED, 7),//
                    new VideoFormat(VideoFormat.IFF_INTERLEAVED_BITMAP, byte[].class, Format.NOT_SPECIFIED, Format.NOT_SPECIFIED, 8),//
                    new VideoFormat(VideoFormat.IFF_INTERLEAVED_BITMAP, byte[].class, Format.NOT_SPECIFIED, Format.NOT_SPECIFIED, 24),//
                });
    }

    @Override
    public void process(Buffer in, Buffer out) {
        if ((in.flags & Buffer.FLAG_DISCARD) != 0) {
            out.flags = Buffer.FLAG_DISCARD;
            return;
        }
        out.format = outputFormat;
        try {
            out.flags = Buffer.FLAG_KEY_FRAME;
            out.duration = in.duration;
            out.timeScale = in.timeScale;

            if (in.data instanceof File) {
                File f = (File) in.data;
                boolean success;
                {
                    InputStream ins = new BufferedInputStream(new FileInputStream(f));
                    try {
                        ILBMDecoder d = new ILBMDecoder(ins);
                        ArrayList<BitmapImage> imgs = d.produceBitmaps();
                        BitmapImage img = imgs.get(0);
                        out.data = img;
                        success = true;
                    } catch (IOException e) {
                        success = false;
                    } finally {
                        ins.close();
                    }
                }
                if (!success) {
                    InputStream ins = new BufferedInputStream(new FileInputStream(f));
                    try {
                        PBMDecoder d = new PBMDecoder(ins);
                        ArrayList<ColorCyclingMemoryImageSource> imgs = d.produce();
                        ColorCyclingMemoryImageSource mis = imgs.get(0);

                        out.data = BitmapImageFactory.toBitmapImage(mis);
                        success = true;
                    } catch (IOException e) {
                        success = false;
                    } finally {
                        ins.close();
                    }
                }
                if (!success) {
                    BufferedImage img = ImageIO.read(f);
                    out.data = BitmapImageFactory.toBitmapImage(img);
                    success = true;
                }
            } else if (in.data instanceof BitmapImage) {
                out.data = in.data;
            } else if (in.data instanceof BufferedImage) {
                out.data = BitmapImageFactory.toBitmapImage((BufferedImage) in.data);
            }
        } catch (IOException e) {
            e.printStackTrace();
            out.flags = Buffer.FLAG_DISCARD;
        }
    }
}
